<?php
// app/Http/Controllers/NotifikasiController.php

namespace App\Http\Controllers;

use App\Models\Notifikasi;
use Illuminate\Http\Request;

class NotifikasiController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }
    
    /**
     * Display user notifications
     */
    public function index()
    {
        $user = auth()->user();
        
        $notifikasi = Notifikasi::where(function($query) use ($user) {
            $query->where('user_id', $user->id)
                  ->orWhere('desa_id', $user->desa_id);
        })
        ->orderBy('created_at', 'desc')
        ->paginate(20);
        
        // Count unread
        $unreadCount = Notifikasi::where(function($query) use ($user) {
            $query->where('user_id', $user->id)
                  ->orWhere('desa_id', $user->desa_id);
        })
        ->where('is_read', false)
        ->count();
        
        return view('dashboard.notifikasi', compact('notifikasi', 'unreadCount'));
    }
    
    /**
     * Mark notification as read
     */
    public function markAsRead($id)
    {
        $notifikasi = Notifikasi::findOrFail($id);
        
        // Check authorization
        if ($notifikasi->user_id !== auth()->id() && 
            $notifikasi->desa_id !== auth()->user()->desa_id) {
            return redirect()->back()->with('error', 'Akses ditolak.');
        }
        
        $notifikasi->update(['is_read' => true]);
        
        return redirect()->back();
    }
    
    /**
     * Mark all as read
     */
    public function markAllAsRead()
    {
        $user = auth()->user();
        
        Notifikasi::where(function($query) use ($user) {
            $query->where('user_id', $user->id)
                  ->orWhere('desa_id', $user->desa_id);
        })
        ->where('is_read', false)
        ->update(['is_read' => true]);
        
        return redirect()->back()->with('success', 'Semua notifikasi ditandai sudah dibaca.');
    }
    
    /**
     * Delete notification
     */
    public function destroy($id)
    {
        $notifikasi = Notifikasi::findOrFail($id);
        
        // Check authorization
        if ($notifikasi->user_id !== auth()->id() && !auth()->user()->isAdmin()) {
            return redirect()->back()->with('error', 'Akses ditolak.');
        }
        
        $notifikasi->delete();
        
        return redirect()->back()->with('success', 'Notifikasi berhasil dihapus!');
    }
    
    /**
     * Get unread count (AJAX)
     */
    public function getUnreadCount()
    {
        $user = auth()->user();
        
        $count = Notifikasi::where(function($query) use ($user) {
            $query->where('user_id', $user->id)
                  ->orWhere('desa_id', $user->desa_id);
        })
        ->where('is_read', false)
        ->count();
        
        return response()->json(['count' => $count]);
    }
    
    /**
     * Get latest notifications (AJAX)
     */
    public function getLatest()
    {
        $user = auth()->user();
        
        $notifikasi = Notifikasi::where(function($query) use ($user) {
            $query->where('user_id', $user->id)
                  ->orWhere('desa_id', $user->desa_id);
        })
        ->orderBy('created_at', 'desc')
        ->take(5)
        ->get();
        
        return response()->json($notifikasi);
    }
}
